// This program solves the time dependent equations for
//   a 2 state system. The diagonal terms have been rotated
//   out so the equations are for the coefficients c.
//
// The calculations have a Gaussian envelope for the E-field
//   amplitude
//
// This program uses the leapfrog algorithm
// psi(t+dt) = psi(t-dt) - 2 i H(t) psi(t)
//

#include <stdio.h>   // standard IO
#include <string.h>  // string manipulations
#include <math.h>    // alternate math functions
#include <complex>   // package for doing complex manipulations
#include <cmath>

// define the type called couble complex
using namespace std ;
typedef complex<double> dcx;

//main program
int main()
{
dcx ceve[2], codd[2] ;
long int numtim, nmod ;
double tau, rabomeg, omeg0, detun, omeg, dt, tstr, tfin, tim, fun, p1, p2 ;

//define the complex number i
const complex <double> icom (0.0,1.0) ;
//define the number pi
const double pi = double(2)*asin(double(1)) ;

//the output will be in whatever is defined here
FILE *output ; output = fopen("simple_32.0_3.0_cpp.txt","w") ;

// the electric field is assumed proportional to exp(-t^2/tau^2) cos(omega t)
tau = 0.1e-6 ;
// define Omega
rabomeg = 32.0*2.0*pi/1.0e-6 ;
// define omega_0
omeg0 = 2.0*pi/1.e-10 ;
// define the detuning
detun = 3.0/tau ;
// define omega
omeg = omeg0 + detun ;
      
// set the time step (you should check convergence with respect to dt)
dt = 0.001e0*2.e0*pi/omeg0 ;
// define the starting (tstr) and final time (tfin)
tstr =-6.0*tau ;
tfin = 6.0*tau ;
// define the number of steps in the numerical integration
numtim = (((tfin-tstr)/dt)) ;
// initialize the coefficients in the two state calculation
ceve[0] = dcx(1.0,0.0) ;
ceve[1] = dcx(0.0,0.0) ;
// these initial conditions for codd assume the E-field is 0 at t=0
codd[0] = ceve[0] ;
codd[1] = ceve[1] ;
      
// nmod is roughly how many steps to skip so the number of output points
//    are the denominator
nmod = numtim/2000 ;
      
// the next loop is the time steps
//  because of how the leapfrog algorithm works, the times are stepped in 2's
 for(long int j = 1 ; j <= numtim ; j+= 2)
 {
// define the time for step j
 tim = tstr + double(j)*dt ;
// compute the Omega X F(t) X cos(omega t)
 fun = rabomeg*exp(-tim*tim/tau/tau)*cos(omeg*tim) ;
// step the psi(t - dt)
 ceve[0] = ceve[0] - 2.0*icom*dt*fun*exp(-icom*omeg0*tim)*codd[1] ;
 ceve[1] = ceve[1] - 2.0*icom*dt*fun*exp( icom*omeg0*tim)*codd[0] ;

// define the time for step j+1
 tim = tstr + double(j+1)*dt ;
// compute the Omega X F(t) X cos(omega t)
 fun = rabomeg*exp(-tim*tim/tau/tau)*cos(omeg*tim) ;
// step the psi(t)
 codd[0] = codd[0] - 2.0*icom*dt*fun*exp(-icom*omeg0*tim)*ceve[1] ;
 codd[1] = codd[1] - 2.0*icom*dt*fun*exp( icom*omeg0*tim)*ceve[0] ;

// every nmod steps output the populations
  if(j%nmod <= 1)
  {
// write the time in microsecs, P_1, P_2, P_1+P_2 to the screen
  p1 = norm(ceve[0]) ;
  p2 = norm(ceve[1]) ;
  printf("%15.7E %15.7E %15.7E %15.7E\n",tim*1.e6,p1,p2,p1+p2) ;
// write the timein microsecs, P_1, P_2, P_1+P_2 to the output file
  fprintf(output,"%15.7E %15.7E %15.7E %15.7E\n",tim*1.e6,p1,p2,p1+p2) ;
  }
 }
// end of time loop
 fclose(output) ;
return 0 ;
}
