// This program solves the time dependent equations for
//   a 2 state system. The diagonal terms have been rotated
//   out so the equations are for the coefficients c.
//
// The calculations have the amplitude for the E-field be constant
//
// The equations use the rotating wave approximation for the c
//
// This program uses the leapfrog algorithm
// psi(t+dt) = psi(t-dt) - 2 i H(t) psi(t)
//

#include <stdio.h>   // standard IO
#include <string.h>  // string manipulations
#include <math.h>    // alternate math functions
#include <complex>   // package for doing complex manipulations
#include <cmath>

// define the type called couble complex
using namespace std ;
typedef complex<double> dcx;

//main program
int main()
{
dcx ceve[2], codd[2], ceve0[2], crotw[2] ;
long int numtim, nmod ;
double tau, rabomeg, omeg0, detun, omeg, dt, tstr, tfin, tim, fun, p1, p2,
       genrab, dmc, dms ;

//define the complex number i
const complex <double> icom (0.0,1.0) ;
//define the number pi
const double pi = double(2)*asin(double(1)) ;

//the output will be in whatever is defined here
FILE *output ; output = fopen("rotwav_const_2.0_3.0_cpp.txt","w") ;

// the electric field is assumed proportional to exp(-t^2/tau^2) cos(omega t)
tau = 0.1e-6 ;
// define Omega
rabomeg = 2.0*2.0*pi/1.0e-6 ;
// define omega_0
omeg0 = 2.0*pi/1.e-10 ;
// define the detuning
detun = 3.0/tau ;
// define omega
omeg = omeg0 + detun ;
// define the general Rabi frequency
genrab = sqrt(detun*detun + rabomeg*rabomeg) ;
      
// set the time step (you should check convergence with respect to dt)
dt = 0.0001e0*2.e0*pi/genrab ;
// define the starting (tstr) and final time (tfin)
tstr = 0.0 ;
tfin = 3.0*2.0*pi/sqrt(detun*detun + rabomeg*rabomeg) ;
// define the number of steps in the numerical integration
numtim = (((tfin-tstr)/dt)) ;
// initialize the coefficients in the two state calculation
ceve0[0] = dcx(1.0,0.0) ;
ceve0[1] = dcx(0.0,0.0) ;
ceve[0] = ceve0[0] ;
ceve[1] = ceve0[1] ;
// these initial conditions for codd assume the E-field constant
//   the sequence is getting codd = c(t + dt) and ceve = c(t) by
//   using 2nd order Runge-Kutta to get the codd
tim = tstr ;
fun = rabomeg*0.5 ;
codd[0] = ceve[0] ;
codd[1] = ceve[1] ;
// ceve in the next step is c(t + dt/2)
ceve[0] = ceve[0] - icom*0.5*dt*fun*exp(-icom*(omeg0-omeg)*tim)*codd[1] ;
ceve[1] = ceve[1] - icom*0.5*dt*fun*exp( icom*(omeg0-omeg)*tim)*codd[0] ;
// codd in the next step is c(t + dt)
tim = tstr+dt*0.5 ;
fun = rabomeg*0.5 ;
codd[0] = codd[0] - icom*dt*fun*exp(-icom*(omeg0-omeg)*tim)*ceve[1] ;
codd[1] = codd[1] - icom*dt*fun*exp( icom*(omeg0-omeg)*tim)*ceve[0] ;
// set ceve to be c(t)
ceve[0] = ceve0[0] ;
ceve[1] = ceve0[1] ;
// finished the set up of the coefficients
      
// nmod is roughly how many steps to skip so the number of output points
//    are the denominator
nmod = numtim/2000 ;
      
// the next loop is the time steps
//  because of how the leapfrog algorithm works, the times are stepped in 2's
 for(long int j = 1 ; j <= numtim ; j+= 2)
 {
// define the time for step j
 tim = tstr + double(j)*dt ;
// compute the Omega X F(t) X cos(omega t)
 fun = rabomeg*0.5 ;
// step the psi(t - dt)
 ceve[0] = ceve[0] - 2.0*icom*dt*fun*exp(-icom*(omeg0-omeg)*tim)*codd[1] ;
 ceve[1] = ceve[1] - 2.0*icom*dt*fun*exp( icom*(omeg0-omeg)*tim)*codd[0] ;

// define the time for step j+1
 tim = tstr + double(j+1)*dt ;
// compute the Omega X F(t) X cos(omega t)
 fun = rabomeg*0.5 ;
// step the psi(t)
 codd[0] = codd[0] - 2.0*icom*dt*fun*exp(-icom*(omeg0-omeg)*tim)*ceve[1] ;
 codd[1] = codd[1] - 2.0*icom*dt*fun*exp( icom*(omeg0-omeg)*tim)*ceve[0] ;

// every nmod steps output the populations
  if(j%nmod <= 1)
  {
// write the time in microsecs, P_1, P_2, P_1+P_2 to the screen
  p1 = norm(ceve[0]) ;
  p2 = norm(ceve[1]) ;
  printf("%15.7E %15.7E %15.7E %15.7E\n",tim*1.e6,p1,p2,p1+p2) ;
// crotw is the rotating wave approximation to the c
  dmc = cos(0.5*genrab*tim) ;
  dms = sin(0.5*genrab*tim) ;
  crotw[0] = ceve0[0]*dcx(dmc,-dms*detun/genrab) -
             ceve0[1]*icom*rabomeg*dms/genrab ;
  crotw[1] = ceve0[1]*dcx(dmc, dms*detun/genrab) -
             ceve0[0]*icom*rabomeg*dms/genrab ;
// write the timein microsecs, P_1, P_2, P_1+P_2, P_1RW, P_2RW to the output file
  fprintf(output,"%15.7E %15.7E %15.7E %15.7E %15.7E %15.7E\n",tim*1.e6,p1,p2,p1+p2,norm(crotw[0]),norm(crotw[1])) ;
  }
 }
// end of time loop
 fclose(output) ;
return 0 ;
}
