/*
 *  First stab at support for metrics in FreeBSD 
 *  by Preston Smith <psmith@physics.purdue.edu>
 *  Wed Feb 27 14:55:33 EST 2002
 *
 */

#include <kvm.h>
#include <fcntl.h>
#include <sys/param.h> 
#include <sys/sysctl.h>
#include <sys/user.h>
#include <sys/dkstat.h>
#include <unistd.h>
#include "ganglia.h"
#include "metric_typedefs.h"

/* Function prototypes */
long percentages(int cnt, int *out, register long *new,
                          register long *old, long *diffs);
 
/*
 * This function is called only once by the gmond.  Use to 
 * initialize data structures, etc or just return SYNAPSE_SUCCESS;
 */
g_val_t
metric_init(void)
{
   g_val_t val;
   val.int32 = SYNAPSE_SUCCESS;
   return val;
}

g_val_t
cpu_num_func ( void )
{
   g_val_t val;
   int ncpu;
   size_t len = sizeof (int);
   if (sysctlbyname("hw.ncpu", &ncpu, &len, NULL, 0) == -1 || !len)
        ncpu = 1;

   val.uint16 = ncpu;
   return val;
}

g_val_t
cpu_speed_func ( void )
{
   g_val_t val;
   int cpu_speed;
   size_t len = sizeof(cpu_speed);
   if (sysctlbyname("machdep.tsc_freq", &cpu_speed, &len, NULL, 0) == -1)
     val.uint16 = 0;

   /* machdep.tsc_freq doesn't seem to always be present. At least on
      my FreeBSD 4 systems. The experts say it gives cpu speed, tho. */
   val.uint16 = cpu_speed /= 1000000;
   return val;
}

g_val_t
mem_total_func ( void )
{
   g_val_t val;
   size_t len;
   int total;
   int mib[2];

   mib[0] = CTL_HW;
   mib[1] = HW_PHYSMEM;
   len = sizeof (total);

   sysctl(mib, 2, &total, &len, NULL, 0);
   total /= 1024;
   val.uint32 = total;
   return val;
}

g_val_t
swap_total_func ( void )
{
   g_val_t val;

   struct kvm_swap swap[1];
   kvm_t *kd;
   int totswap, n;

   kd = kvm_open(NULL, NULL, NULL, O_RDONLY, "kvm_open"); 
   n = kvm_getswapinfo(kd, swap, 1, 0);
   if (n < 0 || swap[0].ksw_total == 0) { 
       val.uint32 = 0;
   }
   totswap = swap[0].ksw_total;
   totswap *= getpagesize() / 1024;
   val.uint32 = totswap;
   kvm_close(kd);

   return val;
}

g_val_t
boottime_func ( void )
{
   g_val_t val;
   struct timeval  boottime;
   int mib[2];
   size_t size;

   mib[0] = CTL_KERN;
   mib[1] = KERN_BOOTTIME;
   size = sizeof(boottime);
   if (sysctl(mib, 2, &boottime, &size, NULL, 0) == -1)
       val.uint32 = 0;

   val.uint32 = boottime.tv_sec;

   return val;
}

g_val_t
sys_clock_func ( void )
{
   g_val_t val;

   val.uint32 = time(NULL);
   return val;
}

g_val_t
machine_type_func ( void )
{
   g_val_t val;
   char machine_type[MAX_G_STRING_SIZE];
   size_t len = MAX_G_STRING_SIZE;
   if (sysctlbyname("hw.machine", &machine_type, &len, NULL, 0) == -1 || !len)
        strncpy( val.str, "x86", MAX_G_STRING_SIZE );

   strncpy( val.str, machine_type, MAX_G_STRING_SIZE );
   return val;
}

g_val_t
os_name_func ( void )
{
   g_val_t val;
   char osname[MAX_G_STRING_SIZE];
   size_t len = MAX_G_STRING_SIZE;
   if (sysctlbyname("kern.ostype", &osname, &len, NULL, 0) == -1 || !len)
        strncpy( val.str, "FreeBSD", MAX_G_STRING_SIZE );

   strncpy( val.str, osname, MAX_G_STRING_SIZE );
 
   return val;
}        

g_val_t
os_release_func ( void )
{
   g_val_t val;
   int mib[2];
   size_t len;
   char *prefix, buf[1024];

   prefix = "";

   mib[0] = CTL_KERN;
   mib[1] = KERN_OSRELEASE;
   len = sizeof(buf);
   if (sysctl(mib, 2, &buf, &len, NULL, 0) == -1)
        strncpy( val.str, "Unknown", MAX_G_STRING_SIZE );

   strncpy( val.str, buf, MAX_G_STRING_SIZE );

   return val;
}        

/* Get the CPU state given by index, from kern.cp_time
 * Use the constants in <sys/dkstat.h>
 * CP_USER=0, CP_NICE=1, CP_SYS=2, CP_INTR=3, CP_IDLE=4
 */
int cpu_state(int which) {

   static long cp_time[CPUSTATES];
   static long cp_old[CPUSTATES];
   static long cp_diff[CPUSTATES];
   static int cpu_states[CPUSTATES];
   static long tot;
   size_t len = sizeof(cp_time);

   /* Copy the last cp_time into cp_old */
   memcpy(&cp_old, &cp_time, CPUSTATES*sizeof(long));
   /* puts kern.cp_time array into cp_time */
   if (sysctlbyname("kern.cp_time", &cp_time, &len, NULL, 0) == -1 || !len)
      return 0.0;
   /* Use percentages function lifted from top(1) to figure percentages */
   tot = percentages(CPUSTATES, cpu_states, cp_time, cp_old, cp_diff);

   return cpu_states[which];
}

g_val_t
cpu_user_func ( void )
{
   g_val_t val;
   int res;

   res = cpu_state(CP_USER); 
   val.f = (float)res/10;
   return val;
}

g_val_t
cpu_nice_func ( void )
{
   g_val_t val;
   int res;

   res = cpu_state(CP_NICE); 
   val.f = (float)res/10;

   return val;
}

g_val_t 
cpu_system_func ( void )
{
   g_val_t val;
   int res;

   res = cpu_state(CP_SYS); 
   val.f = (float)res/10;

   return val;
}

g_val_t 
cpu_idle_func ( void )
{
   g_val_t val;
   int res;

   res = cpu_state(CP_IDLE); 
   val.f = (float)res/10;

   return val;
}

g_val_t 
cpu_aidle_func ( void )
{
   g_val_t val;
   
   val.f = 0.0;
   return val;
}

g_val_t
load_one_func ( void )
{
   g_val_t val;
   double load[3];

   getloadavg(load, 3);
   val.f = load[0];
   return val;
}

g_val_t
load_five_func ( void )
{
   g_val_t val;
   double load[3];

   getloadavg(load, 3);
 
   val.f = load[1];
   return val;
}

g_val_t
load_fifteen_func ( void )
{
   g_val_t val;
   double load[3];

   getloadavg(load, 3);
   val.f = load[2];
   return val;
}

g_val_t
proc_total_func ( void )
{
   g_val_t val;
   int mib[3];
   size_t len;

   mib[0] = CTL_KERN;
   mib[1] = KERN_PROC;
   mib[2] = KERN_PROC_ALL;

   sysctl(mib, 3, NULL, &len, NULL, 0);

   val.uint32 = (len / sizeof (struct kinfo_proc)); 

   return val;
}


/* 
 * Don't know how to do this yet..
 */
g_val_t
proc_run_func( void )
{
   g_val_t val;

   val.uint32 = 0;
   return val;
}

g_val_t
mem_free_func ( void )
{
   g_val_t val;
   size_t len; 
   int free_pages;

   len = sizeof (free_pages);
   if((sysctlbyname("vm.stats.vm.v_free_count", &free_pages, &len, NULL, 0) 
	== -1) || !len) free_pages = 0; 
   free_pages *= getpagesize() / 1024;

   val.uint32 = free_pages;
   return val;
}

g_val_t
mem_shared_func ( void )
{
   g_val_t val;

   val.uint32 = 0;
   return val;
}

g_val_t
mem_buffers_func ( void )
{
   g_val_t val;
   size_t len;
   int buffers;

   len = sizeof (buffers);
   if((sysctlbyname("vfs.bufspace", &buffers, &len, NULL, 0) == -1) || !len)
		buffers = 0; 
   buffers /= 1024;

   val.uint32 = buffers;
   return val;
}

g_val_t
mem_cached_func ( void )
{
   g_val_t val;
   size_t len;
   int cache;

   len = sizeof (cache);
   if((sysctlbyname("vm.stats.vm.v_cache_count", &cache, &len, NULL, 0) == -1) 
	|| !len)
      cache = 0; 

   cache *= getpagesize() / 1024;

   val.uint32 = cache;
   return val;
}

g_val_t
swap_free_func ( void )
{
   g_val_t val;
   struct kvm_swap swap[1];
   kvm_t *kd;
   int totswap, usedswap, freeswap, n;

   kd = kvm_open(NULL, NULL, NULL, O_RDONLY, "kvm_open");
   n = kvm_getswapinfo(kd, swap, 1, 0);
   if (n < 0 || swap[0].ksw_total == 0) {
       val.uint32 = 0;
   }
   totswap = swap[0].ksw_total;
   usedswap = swap[0].ksw_used;
   kvm_close(kd);

   freeswap = totswap-usedswap;
   freeswap *= getpagesize() / 1024;

   val.uint32 = freeswap;
   return val;
}

#include "dnet.h"

static int
find_mtu(const struct intf_entry *entry, void *arg)
{ 
   unsigned int mtu;
   unsigned int *min = (unsigned int *) arg;

   /* Only consider interfaces that are up. */
   if (! entry->intf_flags & INTF_FLAG_UP)
      return 0;

   mtu=entry->intf_mtu;
   if ( !*min || *min>mtu)
      *min=mtu;

   return 0;
}

g_val_t
mtu_func ( void )
{
   /* We want to find the minimum MTU (Max packet size) over all UP interfaces.
*/
   unsigned int min=0;
   g_val_t val;

   intf_t *intf;
   intf = intf_open();
   intf_loop(intf, find_mtu, &min);
   intf_close(intf);
   val.uint32 = min;

   /* A val of 0 means there are no UP interfaces. Shouldn't happen. */
   return val;
}


/*
 * Function to get cpu percentages.
 * Might be changed ever so slightly, but is still mostly:
 * AUTHOR:  Christos Zoulas <christos@ee.cornell.edu>
 *          Steven Wallace  <swallace@freebsd.org>
 *          Wolfram Schneider <wosch@FreeBSD.org>
 *
 * $FreeBSD: src/usr.bin/top/machine.c,v 1.29.2.2 2001/07/31 20:27:05 tmm Exp $
 */

long percentages(int cnt, int *out, register long *new, 
                          register long *old, long *diffs)  {

    register int i;
    register long change;
    register long total_change;
    register long *dp;
    long half_total;

    /* initialization */
    total_change = 0;
    dp = diffs;

    /* calculate changes for each state and the overall change */
    for (i = 0; i < cnt; i++) {
        if ((change = *new - *old) < 0) {
            /* this only happens when the counter wraps */
            change = (int)
                ((unsigned long)*new-(unsigned long)*old);
        }
        total_change += (*dp++ = change);
        *old++ = *new++;
    }
    /* avoid divide by zero potential */
    if (total_change == 0) { total_change = 1; }

    /* calculate percentages based on overall change, rounding up */
    half_total = total_change / 2l;

    /* Do not divide by 0. Causes Floating point exception */
    if(total_change) {
        for (i = 0; i < cnt; i++) {
          *out++ = (int)((*diffs++ * 1000 + half_total) / total_change);
        }
    }

    /* return the total in case the caller wants to use it */
    return(total_change);
}

