/* $Id: server.c,v 1.17 2002/09/11 00:52:03 massie Exp $ */
#include <ganglia.h>
#include <ganglia/barrier.h>
#include <ganglia/gmond_config.h>
#include "dtd.h"
#include "cmdline.h"
#include "metric_typedefs.h"  
#include "node_data_t.h"
#include "key_metrics.h"

extern metric_t metric[];

extern gmond_config_t gmond_config;

/* in gmond.c */
extern g_tcp_socket * server_socket;
extern pthread_mutex_t server_socket_mutex;

/* in listen.c */
extern hash_t *cluster;

typedef struct
   {
      int  fd;
      struct sockaddr_in addr;
      int  valid;
      struct timeval timestamp;      
   }
client_t;

static inline int
buffrd_print( client_t *client, const char *fmt, ... )
{
   int rval, len;
   va_list ap;
   char buf[2048];

   va_start (ap, fmt);

   if(! client->valid ) 
      {
         va_end(ap);
         return 1;
      }

   vsnprintf (buf, sizeof (buf), fmt, ap);  

   len = strlen(buf);

   SYS_CALL( rval, write( client->fd, buf, len)); 
   if ( rval < 0 && rval != len ) 
      {
         va_end(ap);
         client->valid = 0; 
         return 1;
      }

   va_end(ap);
   return 0;
}

static int
data_report ( datum_t *key, datum_t *val, void *arg )
{
   client_t *client = (client_t *)arg;
   return buffrd_print( client, "<METRIC NAME=\"%s\" VAL=\"%s\" TYPE=\"%s\" UNITS=\"%s\" "
                        "TN=\"%ld\" TMAX=\"%ld\" DMAX=\"0\" SLOPE=\"%s\" SOURCE=\"gmond\"/>\n",
                          (char *)key->data,
                          (char *)(((metric_data_t *)val->data)->val),
                          (char *)(((metric_data_t *)val->data)->type),
                          (char *)(((metric_data_t *)val->data)->units),
                          (client->timestamp).tv_sec - (long)(((metric_data_t *)val->data)->timestamp.tv_sec),
                          (unsigned long)(((metric_data_t *)val->data)->tmax),
                          (char *)(((metric_data_t *)val->data)->slope));
}

static int
user_data_report ( datum_t *key, datum_t *val, void *arg )
{
   client_t *client = (client_t *)arg;
   return buffrd_print( client, "<METRIC NAME=\"%s\" VAL=\"%s\" TYPE=\"%s\" UNITS=\"%s\" "
                        "TN=\"%ld\" TMAX=\"%ld\" DMAX=\"0\" SLOPE=\"%s\" SOURCE=\"gmetric\"/>\n",
                       (char *)key->data,
                       (char *)(((metric_data_t *)val->data)->val),
                       (char *)(((metric_data_t *)val->data)->type),
                       (char *)(((metric_data_t *)val->data)->units),
                       (client->timestamp).tv_sec - (long)(((metric_data_t *)val->data)->timestamp.tv_sec),
                       (unsigned long)(((metric_data_t *)val->data)->tmax),
                       (char *)(((metric_data_t *)val->data)->slope));
}

static int
node_report ( datum_t *key, datum_t *val, void *arg )
{
   client_t *client  = (client_t *)arg;

   if (buffrd_print( client, "<HOST NAME=\"%s\" IP=\"%s\" REPORTED=\"%ld\" "
                        "TN=\"%ld\" TMAX=\"%ld\" DMAX=\"0\" LOCATION=\"%s\" GMOND_STARTED=\"%u\">\n",
                  (char *)((node_data_t *)(val->data))->hostname,
                  (char *)key->data,
                  ((node_data_t *)val->data)->timestamp.tv_sec,
                  (client->timestamp).tv_sec - ((node_data_t *)val->data)->timestamp.tv_sec,
                  metric[heartbeat].mcast_max,
                  (char *)((node_data_t *)(val->data))->location,
                  ((node_data_t *)val->data)->start_time))
      return 1;

   if (hash_foreach( ((node_data_t *)(val->data))->hashp, data_report, arg))
      return 1;

   if (hash_foreach( ((node_data_t *)(val->data))->user_hashp, user_data_report, arg))
      return 1;

   if (buffrd_print( client, "</HOST>\n"))
      return 1;
   return 0;
}

void *
server_thread(void *arg)
{
   socklen_t len;
   datum_t host_ip, *hash_rval;
   char remote_ip[MAXHOSTNAMELEN];
   barrier *b = (barrier *)arg;
   client_t client;
   llist_entry *le;

   barrier_barrier(b);

   for (;;)
      {
         len = sizeof(client.addr);
         client.valid = FALSE;

         pthread_mutex_lock  ( &server_socket_mutex );
         SYS_CALL( client.fd, accept(server_socket->sockfd, (struct sockaddr *)&(client.addr), &len));
         pthread_mutex_unlock( &server_socket_mutex );
         if ( client.fd < 0 )
            {
               debug_msg("server_thread() %d clientfd = %d errno=%d\n", pthread_self(), client.fd, errno);
               err_ret("server_thread() accept() error");
               continue;
            }

         debug_msg("server_thread() %d clientfd = %d\n", pthread_self(), client.fd);

         my_inet_ntop( AF_INET, (void *)&(client.addr.sin_addr), remote_ip, MAXHOSTNAMELEN );
         host_ip.data = remote_ip;
         host_ip.size = strlen( remote_ip ) +1;

         if( !strcmp((char *)host_ip.data, "127.0.0.1") )
            {
               client.valid = TRUE;
            }
         else if( gmond_config.all_trusted )
            {
               client.valid = TRUE;
            }
         else 
            {
               /* Search to find if this is a member of the multicast group */ 
               hash_rval = hash_lookup( &host_ip, cluster); 
               if ( hash_rval != NULL )
                  {
                     client.valid = TRUE;
                     datum_free(hash_rval);
                  }
               /* If not a member of multicast group, is it a trusted host? */
               if(! client.valid )
                  {
                     if( llist_search(&(gmond_config.trusted_hosts), (void *)host_ip.data, strcmp, &le) == 0)
                        client.valid = TRUE;
                  }
            }

         if (! client.valid )
            {
               close(client.fd);
               err_msg("server_thread() Host %s tried to connect and was refused", host_ip.data);
               continue;
            }

         /* Send the DTD */
         if(buffrd_print( &client, DTD ))
            {
               close(client.fd);
               continue;
            }

         gettimeofday( &client.timestamp, NULL);

         if(buffrd_print( &client,"<GANGLIA_XML VERSION=\"%s\" SOURCE=\"gmond\">\n", VERSION))
            {
               close(client.fd);
               continue;
            }

         if(buffrd_print( &client,"<CLUSTER NAME=\"%s\" LOCALTIME=\"%ld\" OWNER=\"%s\" LATLONG=\"%s\" URL=\"%s\">\n",
            gmond_config.name, client.timestamp.tv_sec, gmond_config.owner, gmond_config.latlong, gmond_config.url))
            {
               close(client.fd);
               continue;
            }

         if(hash_foreach( cluster, node_report, (void *)&client))
            {
               close(client.fd);
               continue;
            }
         
         buffrd_print( &client, "</CLUSTER>\n</GANGLIA_XML>\n");
         debug_msg("sent data to host %s", (char *)host_ip.data);

         close(client.fd);
      }
   return NULL;
}
