/* $Id */
#include "ganglia.h"
#include "metric_typedefs.h" 
#include "key_metrics.h"

extern metric_t metric[];

#define rand_int(__min,__max) __min + ( __max * rand()/(RAND_MAX+1.0))

extern g_mcast_socket * mcast_socket;
extern pthread_mutex_t mcast_socket_mutex;

void set_check_threshold ( uint32_t key );
void set_mcast_threshold ( uint32_t key );
int set_metric_value ( uint32_t key );

int
mcast_value ( uint32_t key )
{
   XDR xhandle;
   char *p;
   char buf[MAX_MCAST_MSG];
   int rval, len;

   if (! key )
      { 
         /* We don't multicast user-defined data */
         return SYNAPSE_FAILURE;
      }
   debug_msg("mcast_value() mcasting %s value", metric[key].name);

   xdrmem_create(&xhandle, buf, MAX_MCAST_MSG, XDR_ENCODE);
   rval = xdr_u_int(&xhandle, &key);
   if ( rval == 0)
      {
         err_ret("mcast_thread() xdr_u_int() for the key failed");
         return SYNAPSE_FAILURE;
      }
 
   switch ( metric[key].type )
      {
         case g_string:
            p = metric[key].now.str;
            xdr_string(&xhandle, &p, MAX_G_STRING_SIZE);
            break;
         case g_int8:
            xdr_char(&xhandle, &(metric[key].now.int8));
            break;
         case g_uint8:
            xdr_u_char(&xhandle, &(metric[key].now.uint8));
            break;
         case g_int16:
            xdr_short(&xhandle, &(metric[key].now.int16));
            break;
         case g_uint16:
            xdr_u_short(&xhandle, &(metric[key].now.uint16));
            break;
         case g_int32:
            xdr_int(&xhandle, &(metric[key].now.int32));
            break;
         /* This should fall through to the g_uint32 case (no break;) */
         case g_timestamp:
         case g_uint32:
            xdr_u_int(&xhandle, &(metric[key].now.uint32));
            break;
         case g_float:
            xdr_float(&xhandle, &(metric[key].now.f));
            break;
         case g_double:
            xdr_double(&xhandle, &(metric[key].now.d));
            break;
         default:
            err_msg("mcast_thread() got a strange type");
            return SYNAPSE_FAILURE;
      }                                    
   len = xdr_getpos(&xhandle);
 
   debug_msg("encoded %d XDR bytes", len );
 
   pthread_mutex_lock ( &mcast_socket_mutex );
   rval = writen( mcast_socket->sockfd, buf, len );
   pthread_mutex_unlock(&mcast_socket_mutex );
   if ( rval <0)
      {
         err_msg("mcast_thread() error multicasting");
         return SYNAPSE_FAILURE;
      }
 
   debug_msg("XDR data successfully sent");
 
   /* Save this value for threshold comparison later */
   memcpy( &metric[key].last_mcast, &metric[key].now, sizeof(g_val_t)); 

   return SYNAPSE_SUCCESS;
}

void *
monitor_thread ( void *arg )
{
   uint32_t i;
   time_t now;
   int val_threshold_passed, time_threshold_passed;

   /* Set the value for all the configuration metrics ONCE */
   for ( i = 1 ; i < num_key_metrics; i++ )
      { 
         if ( metric[i].check_min <= 0 )
            {
               set_metric_value(i);
               /* We never need to check this metric again..
                  i.e... it will never change */
               metric[i].check_threshold = INT_MAX;
               metric[i].mcast_threshold = 0;
            }   
      }

   for (;;)
      {
         for (i = 1 ; i < num_key_metrics; i++ )
            { 
               val_threshold_passed = FALSE;
               time_threshold_passed = FALSE;

               now = time(NULL);

               /* Check the value threshold first (if it's time to) */
               if ( now > metric[i].check_threshold )
                  {
                     val_threshold_passed = set_metric_value(i);
                     set_check_threshold(i);
                  }

               /* Check the time threshold now */
               if ( now > metric[i].mcast_threshold )
                  {
                     time_threshold_passed = TRUE;
                     set_mcast_threshold(i);
                  }

               /* Multicast data if value or time thresholds are passed */
               if ( val_threshold_passed || time_threshold_passed )
                  {
                     mcast_value(i);
                  }
            }
         sleep(1);
      }
   /* Should never reach this point */
   return NULL;
}

void
set_check_threshold ( uint32_t key )
{
   metric[key].check_threshold = (int)time(NULL) +
               rand_int(metric[key].check_min, metric[key].check_max);
}

void
set_mcast_threshold ( uint32_t key )
{
   metric[key].mcast_threshold = (int)time(NULL) + 
               rand_int(metric[key].mcast_min, metric[key].mcast_max);
}

int 
set_metric_value ( uint32_t key )
{
   int fresh_data;

   debug_msg("set_metric_value() got metric key %d", key);

   if (!key)
      {
         /* We don't monitor user-defined metrics */
         err_msg("check_metric() got key==0");
         return FALSE;
      }

   debug_msg("set_metric_value() exec'd %s_func (%d)", metric[key].name, key);

   metric[key].now = metric[key].func(); 

   fresh_data = FALSE;

   /* Check the value threshold */
   switch( metric[key].type )
      {
         case g_string:
            if ( strcmp (metric[key].now.str, metric[key].last_mcast.str))
               fresh_data = TRUE;
            break;
         case g_int8:
            if ( abs(metric[key].now.int8 - metric[key].last_mcast.int8) 
                 >= metric[key].value_threshold )
               fresh_data = TRUE;
            break;
         case g_uint8:
            if ( abs(metric[key].now.uint8 - metric[key].last_mcast.uint8) 
                 >= metric[key].value_threshold )
               fresh_data = TRUE;
            break;
         case g_int16:
            if ( abs(metric[key].now.int16-metric[key].last_mcast.int16) 
                 >= metric[key].value_threshold )
               fresh_data = TRUE;
            break;
         case g_uint16:
            if ( abs(metric[key].now.uint16-metric[key].last_mcast.uint16) 
                 >= metric[key].value_threshold )
               fresh_data = TRUE;
            break;
         case g_int32:
            if ( abs(metric[key].now.int32 - metric[key].last_mcast.int32) 
                 >= metric[key].value_threshold )
               fresh_data = TRUE;
            break;
         /* This should fall through to the g_uint32 case (no break;) */
         case g_timestamp:
         case g_uint32:
            if ( abs(metric[key].now.uint32-metric[key].last_mcast.uint32) 
                 >= metric[key].value_threshold )
               fresh_data = TRUE;
            break;
         case g_float:
            if ( abs(metric[key].now.f-metric[key].last_mcast.f) 
                 >= metric[key].value_threshold)
               fresh_data = TRUE;
            break;
         case g_double:
            if ( abs(metric[key].now.d-metric[key].last_mcast.d) 
                 >= metric[key].value_threshold)
               fresh_data = TRUE;
            break;
         default:
            err_ret("monitor_thread() received an illegal tag.");
      }                             

   return fresh_data;
}
