/* $Id: listen.c,v 1.10 2002/08/23 22:42:11 sacerdoti Exp $ */
#include <ganglia.h>
#include <ganglia/hash.h>
#include <ganglia/barrier.h>
#include <ganglia/gmond_config.h>
#include "cmdline.h"
#include "key_metrics.h"
#include "metric_typedefs.h"  
#include "node_data_t.h"

extern gmond_config_t gmond_config;

#ifdef AIX
extern void *h_errno_which(void);
#define h_errno   (*(int *)h_errno_which())
#else
extern int h_errno;
#endif

extern metric_t metric[];

extern g_mcast_socket * mcast_join_socket;
extern pthread_mutex_t mcast_join_socket_mutex;

extern hash_t *cluster;

/* in gmond.c */
extern void *send_all_metric_data( void );

/* Will return the memory location of a hash or heartbeat variable */
static void *
pre_process_node( struct sockaddr_in *in, unsigned int key, XDR *xhandle)
{
   int rval;
   datum_t *node_datap, node_ip, *rdatum;
   node_data_t data;
   struct hostent *h;
   void *rtn;
   char remote_ip[16];
   struct timeval tv;
   size_t hash_index;
   bucket_t *bucket;
   uint32_t started = 0;
   char *loc;

   my_inet_ntop( AF_INET, (void *)&(in->sin_addr), remote_ip, 16 );
   debug_msg("%d pre_process_node() remote_ip=%s", (int)pthread_self(), remote_ip);

   node_ip.data = remote_ip; 
   node_ip.size = strlen(remote_ip)+1;

   node_datap = hash_lookup( &node_ip, cluster );

   if ( node_datap == NULL ) 
      {
         /* This is a new host we need to create space for... */
         debug_msg("pre_process_node() received a new node");

         hash_index  = hashval ( &node_ip, cluster );
         WRITE_LOCK( cluster, hash_index ); 
         
         /* We need to make sure that another writing thread didn't beat us here */
         for (bucket = cluster->node[hash_index]->bucket; bucket != NULL; bucket = bucket->next)
            {
               if ( node_ip.size != bucket->key->size )
                  continue;

               if (! strncmp( node_ip.data, bucket->key->data, node_ip.size ))
                  {
                     WRITE_UNLOCK( cluster, hash_index );
                     return NULL; 
                  }
            } 

         data.hostname = g_gethostbyaddr((char *)&(in->sin_addr.s_addr), sizeof(in->sin_addr.s_addr), AF_INET);
         if(!  data.hostname )
            {
               data.hostname = strdup( remote_ip );
            }

         strcpy(data.location,"unspecified");

         gettimeofday(&data.timestamp, NULL);
         debug_msg("pre_process_node() has set the timestamp");

         data.hashp      = hash_create(num_key_metrics);
         if (data.hashp == NULL)
            {
               err_msg("pre_process_node() failed to create hashp");
               WRITE_UNLOCK(cluster, hash_index);
               return NULL;
            }

         debug_msg("pre_process_node() building custom metric hash size=%d",
                    gmond_config.num_custom_metrics);
			data.user_hashp = hash_create(gmond_config.num_custom_metrics);
         if (data.user_hashp == NULL )
            {
               err_msg("pre_process_node() failed to create user_hashp");
               hash_destroy(data.hashp);
               WRITE_UNLOCK(cluster, hash_index);
               return NULL;
            }
			debug_msg("pre_process_node() initialized new hashes");	

         node_datap = datum_new( (char *)&data, sizeof(data) ); 
         if ( node_datap == NULL )
            {
               err_msg("pre_process_node() unable to create a new datum");
               hash_destroy(data.hashp);
               hash_destroy(data.user_hashp);
               WRITE_UNLOCK(cluster, hash_index);
               return NULL;
            }

         WRITE_UNLOCK(cluster, hash_index);
			
         rdatum = hash_insert ( &node_ip, node_datap, cluster );
         if ( rdatum == NULL)
            {
               err_msg("pre_process_node() hash_insert_data() error");
               datum_free(node_datap);
               hash_destroy(data.hashp);
               hash_destroy(data.user_hashp);
               return NULL;
            } 
        
         debug_msg("pre_process_node() inserted data into cluster");

         /* Since they are new.. they will want to know about ALL my data */
         send_all_metric_data();
      }

   debug_msg("pre_process_node() HOSTNAME =%s",
              ((node_data_t *)node_datap->data)->hostname);
   debug_msg("pre_process_node() TIMESTAMP=%d",
              (int)((node_data_t *)node_datap->data)->timestamp.tv_sec);
   debug_msg("pre_process_node() HASHP    =%p",
              ((node_data_t *)node_datap->data)->hashp);
   debug_msg("pre_process_node() USER_HASHP=%p",
              ((node_data_t *)node_datap->data)->user_hashp);

   if( key == heartbeat )
      {
         rtn = NULL;

         rval = xdr_u_int(xhandle, &started);
         if ( rval != 1 )
            {
               err_msg("pre_process_node() failed to get start_time");
               datum_free(node_datap);
               return NULL;
            }

         /* Update the timestamp */
         gettimeofday( &tv, NULL );
         debug_msg("pre_process_node() received an old node");
         memcpy( &(((node_data_t *)(node_datap->data))->timestamp), &tv, sizeof(tv));
         debug_msg("pre_process_node() updated the timestamp");

         /* Check if the remote gmond has been rebooted */
         if ( started != (((node_data_t *)(node_datap->data))->start_time) )
            {
               /* The remote gmond restarted... act on it */
               memcpy( &(((node_data_t *)(node_datap->data))->start_time), &started, sizeof(started));
               send_all_metric_data();
            }

         rdatum = hash_insert ( &node_ip, node_datap, cluster );
         if ( rdatum == NULL)
            {
               err_msg("pre_process_node() hash_insert_data() error");
               datum_free(node_datap);
               return NULL;
            }
      }
   else if ( key == location )
      {
         /* Update the node's location */
         loc = ((node_data_t *)(node_datap->data))->location;
         rval = xdr_string(xhandle, &loc, MAX_G_STRING_SIZE);
         if ( rval != 1 ) {
            err_msg("pre_process_node() failed to get node location (%d)", rval);
            datum_free(node_datap);
            return NULL;
         }
         debug_msg("pre_process_node() updated location: %s", loc);
         rdatum = hash_insert ( &node_ip, node_datap, cluster );
         if (!rdatum) {
            err_msg("pre_process_node() hash_insert_data() error");
            datum_free(node_datap);
            return NULL;
         }
      }
   else if( key )
      {
         rtn = (void *)((node_data_t *)node_datap->data)->hashp;
         debug_msg("pre_process_node() returning the ganglia internal hash pointer %p",
                    rtn);
      }
   else /* key == 0 (user_defined) */
      {
         rtn = (void *)((node_data_t *)node_datap->data)->user_hashp;
         debug_msg("pre_process_node() RETURNING USER DATA HASH POINTER %p",
                    rtn);
      }

   datum_free(node_datap);
   return rtn;
}

static void
strip_quotes( char *str, uint32_t len )
{
   uint32_t i;
 
   for (i=0; i<len; i++)
      {
         if ( str[i] == '"' )
            str[i] = ' ';
      }
}
   
void *
mcast_listen_thread(void *arg)
{
   int rval;
   char buf[BUFSIZ];
   uint32_t key;
   socklen_t len;
   struct sockaddr_in addr;
   char *p;
   XDR xhandle;
   g_val_t mcast_val;
   hash_t *hashp;
   datum_t metricname, metricval, *rdatum;
   metric_data_t metricdata;
   uint32_t datalen;
   char name[32];
   barrier *b = (barrier *)arg;

   barrier_barrier(b);

   debug_msg("mcast_listen_thread() started %d", (int)pthread_self());

   for (;;)
      {
         len = sizeof ( addr );

         pthread_mutex_lock  (&mcast_join_socket_mutex);
         rval = recvfrom( mcast_join_socket->sockfd, (void *)buf, MAX_MCAST_MSG,
                          0, (struct sockaddr *)&(addr), &len);
         pthread_mutex_unlock(&mcast_join_socket_mutex);
         if ( rval < 0 )
            {
               err_ret("mcast_listen_thread() recvfrom() error");
               pthread_mutex_unlock(&mcast_join_socket_mutex);
               continue;
            }

         debug_msg("mcast_listen_thread() got a %d byte multicast message", rval);

         xdrmem_create(&xhandle, buf, sizeof(buf)-1, XDR_DECODE);

         rval = xdr_u_int(&xhandle, &key);
         if ( rval == 0 )
            {
               err_ret("mcast_listen_thread() xdr_u_int() for key failure");
               continue;
            }

         debug_msg("mcast_listen_thread() received key %d", key );

         if (key > (num_key_metrics-1))
            {
               debug_msg("Got a key with a value greater than the largest metric key");
               continue;
            }

         debug_msg("mcast_listen_thread() received metric data %s", metric[key].name );

         if( key == heartbeat )
            {
               hashp = (hash_t *) pre_process_node( &addr, key, &xhandle );

               if(hashp)
                  {

                  }

               continue;
            }
         else if ( key == location )
            {
               /* Update the node_data_t->location attribute. Don't add as a normal metric. */
               hashp = (hash_t *) pre_process_node( &addr, key, &xhandle );
               continue;
            }
         else if (key)
            {
               hashp = (hash_t *) pre_process_node( &addr, key, &xhandle );
               if ( hashp == NULL )
                  {
                     continue;
                  } 
               debug_msg("mcast_listen_thread() got internal hash %p",hashp);

               strncpy(name, metric[key].name, MAX_G_STRING_SIZE);
               metricname.data  = name;
               metricname.size = strlen( name ) + 1;

               gettimeofday( &metricdata.timestamp, NULL);
               metricdata.tmax = metric[key].mcast_max;
               if( metric[key].check_min < 0 )
                  strcpy(metricdata.slope, "zero");
               else
                  strcpy(metricdata.slope, "both");

               switch( metric[key].type )
                  {
                     case g_string:
                        p = mcast_val.str;
                        rval = xdr_string(&xhandle, &p,  MAX_G_STRING_SIZE);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_string() error");
                              continue;
                           }
                        snprintf(metricdata.val, FRAMESIZE,
                                 metric[key].fmt, mcast_val.str);
                        strncpy(metricdata.type, "string", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_int8:
                        rval = xdr_char(&xhandle, &mcast_val.int8);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_char() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.int8);
                        strncpy(metricdata.type, "int8", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_uint8:
                        rval = xdr_u_char(&xhandle, &mcast_val.uint8);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_u_char() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.uint8);
                        strncpy(metricdata.type, "uint8", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_int16:
                        rval = xdr_short(&xhandle, &mcast_val.int16);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_short() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.int16);
                        strncpy(metricdata.type, "int16", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_uint16:
                        rval = xdr_u_short(&xhandle, &mcast_val.uint16);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_u_short() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.uint16);
                        strncpy(metricdata.type, "uint16", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_int32:
                        rval = xdr_int(&xhandle, &mcast_val.int32);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_int() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.int32);
                        strncpy(metricdata.type, "int32", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                    case g_uint32:
                        rval = xdr_u_int(&xhandle, &mcast_val.uint32);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_u_int() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.uint32);
                        strncpy(metricdata.type, "uint32", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                    /* A timestamp type is always a uint32 (32-bit unsigned int) */
                    case g_timestamp:
                        rval = xdr_u_int(&xhandle, &mcast_val.uint32);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_u_int() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.uint32);
                        strncpy(metricdata.type, "timestamp", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_float:
                        rval = xdr_float(&xhandle, &mcast_val.f);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_float() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.f);
                        strncpy(metricdata.type, "float", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     case g_double:
                        rval = xdr_double(&xhandle, &mcast_val.d);
                        if ( rval != 1 )
                           {
                              err_ret("mcast_listen_thread() xdr_double() error");
                              continue;
                           }
                        snprintf(metricdata.val,  FRAMESIZE,
                                 metric[key].fmt, mcast_val.d);
                        strncpy(metricdata.type, "double", MAX_TYPE_LEN);
                        strncpy(metricdata.units, metric[key].units, MAX_UNITS_LEN);
                        break;
                     default:
                        err_msg("mcast_listen_thread() error: STRANGE type!");
                        continue;
                  }
                  /* Find the size of the value string, including the null terminating char. */
                  metricdata.valsize = strlen(metricdata.val) + 1;
            }
         else  /* key = 0 user_defined */
            {
               int slope;

               /* Values for TN and TMAX */
               gettimeofday( &metricdata.timestamp, NULL);
               /* For now */
               metricdata.tmax = 0; 

               hashp = (hash_t *)pre_process_node( &addr, key, &xhandle);
               if ( hashp == NULL )
                  {
                     continue;
                  }    
               debug_msg("mcast_listen_thread() got USER-DEFINED hash %p", hashp);

               p = metricdata.type;
               rval = xdr_bytes(&xhandle, &p, &datalen, MAX_TYPE_LEN);
               if ( rval == 0 )
                  { 
                     err_msg("mcast_listen_thread() xdr_bytes for type error");
                     continue;
                  }
               strip_quotes(p, datalen);

               p = metricname.data;
               rval = xdr_bytes(&xhandle, &p, &(metricname.size), MAX_MCAST_MSG);
               if ( rval == 0 )
                  {
                     err_msg("mcast_listen_thread() xdr_bytes for name error");
                     continue;
                  }
               strip_quotes(p, metricname.size);
               debug_msg("mcast_listen_thread() name = %s", p);

               p = metricdata.val;
               rval = xdr_bytes(&xhandle, &p, &datalen, FRAMESIZE);
               if ( rval == 0 )
                  {
                     err_msg("mcast_listen_thread() xdr_bytes for val error");
                     continue;
                  }
               strip_quotes(p, datalen);
               metricdata.valsize=datalen;

               p = metricdata.units;
               rval = xdr_bytes(&xhandle, &p, &datalen, MAX_UNITS_LEN);
               if ( rval == 0 )
                  {
                     err_msg("mcast_listen_thread() xdr_bytes for units error");
                     continue;
                  }   
               strip_quotes(p, datalen);

               rval = xdr_u_int(&xhandle, &slope);
               if ( rval == 0 )
                  {
                     err_msg("mcast_listen_thread() xdr_bytes for slope error");
                     continue;
                  }
 
               switch(slope)
                  {
                     case 0: strcpy(metricdata.slope, "zero");
                        break;
                     case 1: strcpy(metricdata.slope, "positive");
                        break;
                     case 2: strcpy(metricdata.slope, "negative");
                        break;
                     default: strcpy(metricdata.slope, "both");
                  }

               rval = xdr_u_int(&xhandle, &(metricdata.tmax));
               if ( rval == 0 )
                  {
                     err_msg("mcast_listen_thread() xdr_bytes for TMAX error");
                     continue;
                  }

               debug_msg("user-defined data: type=%s name=%s val=%s (%d) units=%s",
                  metricdata.type,metricname.data,metricdata.val,metricdata.valsize,metricdata.units);

               /*
               for (rval = 0; rval < 25; rval++)
                  debug_msg("special: key=%d val=%s", rval, metric[rval].name);
               */
            }

            debug_msg("mcast_listen_thread() built metricdata struct");

            metricval.data = &metricdata;
            /* Careful here not to waste memory. Assumes sizeof() returns size
               in bytes, and one char is one byte. */
            metricval.size = sizeof(metricdata) - FRAMESIZE + metricdata.valsize;

            debug_msg("mcast_listen_thread() attempting to hash_insert_data");
            rdatum = hash_insert(&metricname, &metricval, hashp);
            if ( rdatum == NULL )
               {
                  err_msg("mcast_listen_thread() hash_data_insert() error");
               }
            else
               {
                  debug_msg("mcast_listen_thread() inserted data into %p",hashp);
               }

      } /* for (;;) */
   return NULL;
}
